<?php
require_once('../includes/admin_check.php'); // Solo admin
require_once('../config/database.php');

$upload_message = '';
$error_message = '';
$success_count = 0;
$error_count = 0;
$errors_details = []; // Para guardar detalles de errores por fila

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_FILES["user_csv"])) {

    $file = $_FILES["user_csv"];

    // 1. Validaciones básicas del archivo subido
    if ($file["error"] !== UPLOAD_ERR_OK) {
        $error_message = "Error al subir el archivo (Código: " . $file["error"] . ")";
    } elseif ($file["type"] != "text/csv" && $file["type"] != "application/vnd.ms-excel") { // Validar tipo MIME básico
         $error_message = "Tipo de archivo inválido. Solo se permiten archivos CSV.";
    } elseif ($file["size"] > 5000000) { // Límite de 5MB (ajustar)
        $error_message = "El archivo es demasiado grande (Máx 5MB).";
    } else {
        // Mover archivo temporal a una ubicación procesable
        $tmp_name = $file["tmp_name"];
        $row = 1; // Contador de filas para mensajes de error
        $header = null; // Para guardar la cabecera del CSV

        if (($handle = fopen($tmp_name, "r")) !== FALSE) {
            $conn->begin_transaction(); // Iniciar transacción para procesar todo o nada

            try {
                while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) { // Leer CSV (coma como separador)
                    if ($row === 1) {
                        // Validar cabecera (ajusta nombres si tu CSV usa otros)
                        $header = array_map('trim', $data);
                        $expected_header = ['codigo', 'nombre_completo', 'username', 'password'];
                        if ($header !== $expected_header) {
                            throw new Exception("Cabecera del CSV inválida. Se esperaba: 'codigo, nombre_completo, username, password'");
                        }
                        $row++;
                        continue; // Saltar la fila de cabecera
                    }

                    // Asegurarse que hay 4 columnas
                    if (count($data) !== 4) {
                        $errors_details[] = "Fila $row: Número incorrecto de columnas.";
                        $error_count++;
                        $row++;
                        continue; // Saltar esta fila
                    }

                    // Asignar datos (limpiando espacios)
                    $codigo = trim($data[0]);
                    $nombre_completo = trim($data[1]);
                    $username = trim($data[2]);
                    $password = trim($data[3]);

                    // Validación básica de datos (¡mejorar!)
                    if (empty($codigo) || empty($nombre_completo) || empty($username) || empty($password)) {
                         $errors_details[] = "Fila $row: Faltan datos (Usuario: $username, Código: $codigo).";
                         $error_count++;
                         $row++;
                         continue;
                    }

                    // Hashear contraseña
                    $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                    if ($hashed_password === false) {
                         $errors_details[] = "Fila $row: Error al hashear password (Usuario: $username).";
                         $error_count++;
                         $row++;
                         continue;
                    }

                    // Preparar INSERT (Evitar duplicados si es necesario - requiere consulta previa o manejo de error de duplicado)
                    // Por ahora, asumimos que no existen o fallará por constraint UNIQUE
                    $sql = "INSERT INTO user (employee_id, full_name, username, password_hash, is_admin) VALUES (?, ?, ?, ?, 0)"; // is_admin = 0 por defecto
                    $stmt = $conn->prepare($sql);

                    if ($stmt) {
                        $stmt->bind_param("ssss", $codigo, $nombre_completo, $username, $hashed_password);
                        if ($stmt->execute()) {
                            $success_count++;
                        } else {
                            // Capturar error (ej. duplicado)
                            $errors_details[] = "Fila $row: Error DB (" . $stmt->errno . ") " . $stmt->error . " (Usuario: $username, Código: $codigo).";
                            $error_count++;
                        }
                        $stmt->close();
                    } else {
                         $errors_details[] = "Fila $row: Error preparando statement DB.";
                         $error_count++;
                    }
                    $row++;
                } // Fin while

                fclose($handle);

                // Decidir si confirmar o deshacer la transacción
                if ($error_count > 0) {
                    $conn->rollback(); // Deshacer todo si hubo errores
                    $error_message = "Proceso fallido. No se importó ningún usuario debido a errores. Ver detalles abajo.";
                } else {
                    $conn->commit(); // Confirmar todo si no hubo errores
                    $upload_message = "¡Carga masiva completada! $success_count usuarios importados.";
                }

            } catch (Exception $e) {
                // Capturar excepciones generales (ej. cabecera inválida)
                $conn->rollback(); // Asegurar rollback
                $error_message = "Error durante el proceso: " . $e->getMessage();
                if (isset($handle) && is_resource($handle)) fclose($handle);
            }

        } else {
            $error_message = "No se pudo abrir el archivo CSV subido.";
        }
    }
    $conn->close(); // Cerrar conexión
}

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Admin - Carga Masiva de Usuarios</title>
    <link rel="stylesheet" href="../public/css/style.css"> </head>
<body>
    <div class="container">
        <div class="header">
            <h1>Carga Masiva de Usuarios (CSV)</h1>
            <p><a href="users_list.php">Volver a la Lista</a> | <a href="../logout.php">Cerrar Sesión</a></p>
        </div>

        <?php if ($upload_message): ?>
            <p class="success"><?php echo htmlspecialchars($upload_message); ?></p>
        <?php endif; ?>
        <?php if ($error_message): ?>
            <p class="error"><?php echo htmlspecialchars($error_message); ?></p>
        <?php endif; ?>
        <?php if (!empty($errors_details)): ?>
            <div class="error-details">
                <h4>Detalles de Errores (<?php echo $error_count; ?> filas afectadas):</h4>
                <ul>
                    <?php foreach ($errors_details as $detail): ?>
                        <li><?php echo htmlspecialchars($detail); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>


        <form action="bulk_upload.php" method="post" enctype="multipart/form-data">
            <p>Selecciona un archivo CSV para subir. Debe tener las columnas: <strong>codigo, nombre_completo, username, password</strong></p>
            <div>
                <label for="user_csv">Archivo CSV:</label>
                <input type="file" name="user_csv" id="user_csv" accept=".csv, text/csv" required>
            </div>
            <div>
                <button type="submit">Subir e Importar Usuarios</button>
            </div>
        </form>
         <p><small>Nota: Si un 'codigo' (ID Empleado) o 'username' ya existe, esa fila dará error y no se importará ningún usuario en este lote.</small></p>

    </div>
</body>
</html>