<?php
// supervisor/upload_receipts.php

require_once('../includes/supervisor_check.php'); 
require_once('../includes/functions.php'); 

$page_title = 'Supervisor - Cargar Recibos de Nómina';

// ... (TODA LA LÓGICA PHP DE PROCESAMIENTO DEL POST SE MANTIENE EXACTAMENTE IGUAL QUE ANTES) ...
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_FILES["receipts_zip_file"])) {
    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        $_SESSION['sv_upload_error_message'] = "Error de validación de seguridad (CSRF). Inténtelo de nuevo.";
        header("Location: upload_receipts.php");
        exit;
    }
    $file = $_FILES["receipts_zip_file"];
    $target_dir = __DIR__ . "/../recibos_nomina/"; 
    $upload_errors = [];
    if ($file['error'] !== UPLOAD_ERR_OK) { $upload_errors[] = "Error al subir el archivo (Código: " . $file['error'] . ")."; }
    $file_type = mime_content_type($file['tmp_name']);
    if ($file_type !== 'application/zip') { $upload_errors[] = "Tipo de archivo inválido. Solo se permiten archivos .zip."; }
    if ($file['size'] > 209715200) { $upload_errors[] = "El archivo es demasiado grande (Máximo 200MB)."; }
    if (empty($upload_errors)) {
        if (!is_dir($target_dir)) { mkdir($target_dir, 0775, true); }
        if (!is_writable($target_dir)) {
            $_SESSION['sv_upload_error_message'] = "Error crítico: El directorio '{$target_dir}' no tiene permisos de escritura.";
            header("Location: upload_receipts.php");
            exit;
        }
        $zip = new ZipArchive;
        if ($zip->open($file['tmp_name']) === TRUE) {
            $extracted_count = 0;
            $extraction_errors = [];
            for ($i = 0; $i < $zip->numFiles; $i++) {
                $filename_in_zip = $zip->getNameIndex($i);
                if (substr($filename_in_zip, -1) === '/' || strpos($filename_in_zip, '__MACOSX') !== false) { continue; }
                if (strpos($filename_in_zip, '..') !== false) { $extraction_errors[] = "Se omitió el archivo con ruta peligrosa: " . htmlspecialchars($filename_in_zip); continue; }
                $base_filename = basename($filename_in_zip);
                $file_content = $zip->getFromIndex($i);
                if ($file_content === false) { $extraction_errors[] = "No se pudo leer el contenido del archivo '" . htmlspecialchars($filename_in_zip) . "' dentro del zip."; continue; }
                $destination_path = $target_dir . $base_filename;
                if (file_put_contents($destination_path, $file_content)) {
                    $extracted_count++;
                } else { $extraction_errors[] = "No se pudo escribir el archivo '" . htmlspecialchars($base_filename) . "' en el directorio de destino."; }
            }
            $zip->close();
            if ($extracted_count > 0) { $_SESSION['sv_upload_success_message'] = "¡Éxito! Se han extraído y colocado {$extracted_count} archivos de recibos en el sistema."; }
            if (!empty($extraction_errors)) { $_SESSION['sv_upload_error_message'] = ($_SESSION['sv_upload_error_message'] ?? '') . "<br><strong>Problemas durante la extracción:</strong><br>" . implode("<br>", $extraction_errors); }
            if ($extracted_count == 0 && empty($extraction_errors) && empty($_SESSION['sv_upload_error_message'])) { $_SESSION['sv_upload_error_message'] = "El archivo .zip no contenía archivos válidos para extraer."; }
        } else { $_SESSION['sv_upload_error_message'] = "No se pudo abrir el archivo .zip. Puede que esté corrupto."; }
    } else { $_SESSION['sv_upload_error_message'] = implode("<br>", $upload_errors); }
    header("Location: upload_receipts.php");
    exit;
}

$csrf_token = generate_csrf_token();
require_once('../includes/header.php');
?>
<main class="container">
    <h2><?php echo htmlspecialchars($page_title); ?></h2>
    <p><a href="dashboard.php" class="button button-outline">&laquo; Volver al Dashboard de Supervisor</a></p>

    <?php // Mostrar mensajes de sesión específicos para el supervisor
    if (isset($_SESSION['sv_upload_success_message'])): ?>
        <p class="success"><?php echo $_SESSION['sv_upload_success_message']; unset($_SESSION['sv_upload_success_message']); ?></p>
    <?php endif; ?>
    <?php if (isset($_SESSION['sv_upload_error_message'])): ?>
        <p class="error"><?php echo $_SESSION['sv_upload_error_message']; unset($_SESSION['sv_upload_error_message']); ?></p>
    <?php endif; ?>

    <form action="upload_receipts.php" method="post" enctype="multipart/form-data">
        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
        
        <h3>Subir Archivo .zip con Recibos de Nómina</h3>
        
        <div class="form-group">
            <label for="receipts_zip_file">Archivo .zip de Recibos:</label>
            <input type="file" name="receipts_zip_file" id="receipts_zip_file" accept=".zip,application/zip" required>
        </div>
        
        <div class="form-group" style="margin-top: 15px;">
            <button type="submit" class="button">Subir y Extraer Recibos</button>
        </div>
    </form>
    
    <div style="margin-top: 2rem; padding: 1rem; border: 1px solid #ccc; border-radius: 5px; background-color: #f8f9fa;">
        <h4>Instrucciones Importantes</h4>
        <ul>
            <li>El archivo a subir debe ser de tipo `.zip`.</li>
            <li>El tamaño máximo del archivo es de 200MB.</li>
            <li>Para una carga correcta, se recomienda que **los recibos de nómina estén dentro de una carpeta**, y que luego esa carpeta sea la que se comprima. Por ejemplo: `Mis Documentos > Recibos Mayo 2025 > [aquí todos los recibos]`. Luego, haz clic derecho sobre la carpeta `Recibos Mayo 2025` y elige "Comprimir en archivo .zip".</li>
            <li>El sistema está diseñado para extraer los archivos de recibos sin importar si están dentro de una carpeta o no en el `.zip`.</li>
            <li>Asegúrate de que los nombres de los archivos de recibos sigan el formato estándar de la aplicación.</li>
        </ul>
    </div>
    </main>
<?php
require_once('../includes/footer.php');
?>