<?php
// reset_password.php

if (session_status() === PHP_SESSION_NONE) {
    session_start([
        'cookie_httponly' => true,
        'cookie_secure' => isset($_SERVER['HTTPS']),
        'cookie_samesite' => 'Lax'
    ]);
}

require_once('config/database.php'); 
require_once('includes/functions.php'); 

$page_title = 'Establecer Nueva Contraseña';
$token_is_valid = false;
$user_id_to_reset = null;
$error_message = '';
$plain_token_from_url = '';

// --- 1. VERIFICACIÓN DEL TOKEN (cuando la página carga con GET) ---
if ($_SERVER["REQUEST_METHOD"] == "GET" && isset($_GET['token'])) {
    $plain_token_from_url = $_GET['token'];

    if (empty($plain_token_from_url) || !ctype_xdigit($plain_token_from_url)) {
        // El token debe ser una cadena hexadecimal, si no, es inválido.
        header("Location: login.php?error=invalid_token");
        exit;
    } else {
        // Buscar todos los tokens que no han expirado para compararlos
        $sql_find_token = "SELECT id, password_reset_token FROM user WHERE password_reset_expires > NOW() AND password_reset_token IS NOT NULL";
        $result = $conn->query($sql_find_token);

        if ($result && $result->num_rows > 0) {
            while ($user_row = $result->fetch_assoc()) {
                if (password_verify($plain_token_from_url, $user_row['password_reset_token'])) {
                    $token_is_valid = true;
                    $user_id_to_reset = $user_row['id'];
                    // Guardar los datos necesarios en la sesión para el envío del formulario POST
                    $_SESSION['password_reset_user_id'] = $user_id_to_reset;
                    $_SESSION['password_reset_token_validated'] = true; // Flag para el POST
                    break; 
                }
            }
        }
        
        if (!$token_is_valid) {
            header("Location: login.php?error=invalid_token");
            exit;
        }
    }
}

// --- 2. PROCESAMIENTO DEL FORMULARIO DE NUEVA CONTRASEÑA (cuando se envía con POST) ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Verificar que el usuario pasó por la validación del token en el paso GET
    if (!isset($_SESSION['password_reset_user_id']) || !isset($_SESSION['password_reset_token_validated']) || $_SESSION['password_reset_token_validated'] !== true) {
        header("Location: login.php?error=invalid_token");
        exit;
    }

    $user_id_to_reset = $_SESSION['password_reset_user_id'];
    $token_is_valid = true; // Si llegamos aquí, el token era válido, mantenemos el formulario visible si hay errores POST

    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        $error_message = "Error de validación de seguridad. Por favor, intente de nuevo.";
    } else {
        $new_password = $_POST['new_password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';

        $validation_errors = [];
        if (empty($new_password) || empty($confirm_password)) {
            $validation_errors[] = "Ambos campos de contraseña son obligatorios.";
        } elseif ($new_password !== $confirm_password) {
            $validation_errors[] = "Las contraseñas no coinciden.";
        } else {
            // Función de validación de complejidad (debe estar en functions.php)
            if (function_exists('validarPasswordConRequisitos')) {
                 $complexity_errors = validarPasswordConRequisitos($new_password);
            } else { // Fallback por si la función no se definió
                 if (strlen($new_password) < 8) $complexity_errors[] = "Mínimo 8 caracteres.";
                 if (!preg_match('/[A-Z]/', $new_password)) $complexity_errors[] = "Requiere al menos una mayúscula.";
                 if (!preg_match('/[a-z]/', $new_password)) $complexity_errors[] = "Requiere al menos una minúscula.";
                 if (!preg_match('/[0-9]/', $new_password)) $complexity_errors[] = "Requiere al menos un número.";
            }
            if (!empty($complexity_errors)) {
                $validation_errors = array_merge($validation_errors, $complexity_errors);
            }
        }

        if (empty($validation_errors)) {
            $new_password_hash = password_hash($new_password, PASSWORD_DEFAULT);
            
            // Actualizar contraseña, poner force_change a 0, y limpiar token de reseteo
            $sql_update = "UPDATE user SET password_hash = ?, force_password_change = 0, password_reset_token = NULL, password_reset_expires = NULL WHERE id = ?";
            $stmt_update = $conn->prepare($sql_update);
            if ($stmt_update) {
                $stmt_update->bind_param("si", $new_password_hash, $user_id_to_reset);
                if ($stmt_update->execute()) {
                    // Limpiar sesión y redirigir a login con mensaje de éxito
                    session_unset(); session_destroy(); 
                    header("Location: login.php?message=password_reset_success");
                    exit;
                } else {
                    $error_message = "Error al actualizar su contraseña. Por favor, inténtelo de nuevo.";
                    error_log("RESET_PASSWORD: Error ejecutando update final: " . $stmt_update->error);
                }
                $stmt_update->close();
            } else {
                $error_message = "Error del servidor. Por favor, inténtelo más tarde.";
                error_log("RESET_PASSWORD: Error preparando update final: " . $conn->error);
            }
        } else {
            $error_message = implode("<br>", array_map('htmlspecialchars', $validation_errors));
        }
    }
    if(isset($conn) && $conn instanceof mysqli) $conn->close();
}

// Redirigir si se accede directamente sin token
if ($_SERVER["REQUEST_METHOD"] !== "POST" && !$token_is_valid) {
    header("Location: login.php");
    exit;
}


$csrf_token = generate_csrf_token();
require_once('includes/header.php'); 
?>
<main class="container" style="max-width: 600px;">
    <h2><?php echo htmlspecialchars($page_title); ?></h2>

    <?php if ($token_is_valid): ?>
        <?php if ($error_message): ?>
            <p class="error"><?php echo $error_message; ?></p>
        <?php endif; ?>
        
        <p>Hola de nuevo. Por favor, establece tu nueva contraseña.</p>

        <form action="reset_password.php" method="POST">
            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
            
            <div class="form-group">
                <label for="new_password">Nueva Contraseña:</label>
                <input type="password" id="new_password" name="new_password" required>
            </div>
            <div class="form-group">
                <label for="confirm_password">Confirmar Nueva Contraseña:</label>
                <input type="password" id="confirm_password" name="confirm_password" required>
            </div>
            <div class="form-group">
                <button type="submit" class="button">Guardar Nueva Contraseña</button>
            </div>
        </form>
    <?php else: ?>
        <p class="error">El enlace para restablecer la contraseña no es válido o ha expirado. Por favor, solicita uno nuevo.</p>
        <p><a href="forgot_password.php" class="button">Solicitar Nuevo Enlace</a></p>
    <?php endif; ?>

</main>
<?php
require_once('includes/footer.php'); 
if (isset($conn) && $conn instanceof mysqli && $conn->ping()) {
    $conn->close();
}
?>