<?php
// procesar_aceptacion_y_descarga.php

// 1. INCLUDES Y CONFIGURACIÓN INICIAL
require_once('includes/auth_check.php'); 
require_once('config/database.php');    
require_once('includes/functions.php'); // Para CSRF y otras
require_once('includes/fpdf/fpdf.php');   

$nombre_empresa_cliente = "CORPORATIVO EMPRESARIAL"; 
$ruta_logo_cliente_web = "public/images/logo_corporativo.png"; 
$ruta_logo_cliente_fpdf = __DIR__ . '/' . $ruta_logo_cliente_web; 

// 2. VERIFICACIÓN Y RECEPCIÓN DE DATOS DEL FORMULARIO (POST)
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['error_message'] = "Acceso no permitido a esta página.";
    header("Location: dashboard.php");
    exit;
}

// VALIDAR TOKEN CSRF PRIMERO
if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
    $_SESSION['error_message'] = "Error de validación de seguridad (CSRF). La acción no se realizó. Por favor, intente de nuevo desde la página anterior.";
    error_log("CSRF token validation failed for procesar_aceptacion_y_descarga.php. User ID: " . ($_SESSION['user_id'] ?? 'N/A'));
    header("Location: dashboard.php"); 
    exit;
}

// Si el token es válido, continuar...
$tipo_descarga = $_POST['tipo_descarga'] ?? null;
$archivos_a_firmar_json = $_POST['archivos_a_firmar_json'] ?? null;
$param_descarga_filename = null;
$param_descarga_filter_month_from = null;
$param_descarga_filter_month_to = null;

if ($tipo_descarga === 'individual') {
    $param_descarga_filename = $_POST['param_descarga_filename'] ?? null;
} elseif ($tipo_descarga === 'masiva') {
    $param_descarga_filter_month_from = $_POST['param_descarga_filter_month_from'] ?? null;
    $param_descarga_filter_month_to = $_POST['param_descarga_filter_month_to'] ?? null;
}

// ... (El resto de tu código de procesar_aceptacion_y_descarga.php se mantiene igual que la última versión que te di) ...
// (Desde la validación básica de datos recibidos, datos del empleado, configuración de directorios,
//  el bucle foreach para generar PDF y registrar en BD, y la sección final de manejo de errores y
//  configuración de descarga en sesión antes de redirigir a dashboard.php)
// Asegúrate de que toda esa lógica esté aquí.
// Lo que sigue es la continuación de tu código original, adaptado:

if (empty($tipo_descarga) || empty($archivos_a_firmar_json)) {
    $_SESSION['error_message'] = "Faltan datos necesarios para procesar la aceptación.";
    error_log("PROCESAR_ACEPTACION: Datos POST incompletos. Tipo: {$tipo_descarga}, JSON: {$archivos_a_firmar_json}");
    header("Location: dashboard.php");
    exit;
}
$archivos_nombres_recibos = json_decode($archivos_a_firmar_json, true);
if (json_last_error() !== JSON_ERROR_NONE || !is_array($archivos_nombres_recibos) || empty($archivos_nombres_recibos)) {
    $_SESSION['error_message'] = "Error en el formato de los archivos a firmar.";
    error_log("PROCESAR_ACEPTACION: Error decodificando JSON de archivos: " . json_last_error_msg() . " - Data: " . $archivos_a_firmar_json);
    header("Location: dashboard.php");
    exit;
}
$user_id_actual = $_SESSION['user_id'] ?? null;
$nombre_empleado = $_SESSION['full_name'] ?? 'Empleado Desconocido';
$id_empleado_nomina_original = $_SESSION['employee_id'] ?? null; 
if (!$user_id_actual || !$id_empleado_nomina_original) {
    $_SESSION['error_message'] = "Su sesión ha expirado o es inválida. Por favor, inicie sesión nuevamente.";
    error_log("PROCESAR_ACEPTACION: Falta user_id ({$user_id_actual}) o employee_id ({$id_empleado_nomina_original}) en sesión.");
    header("Location: login.php");
    exit;
}

$directorio_cartas = __DIR__ . '/cartas_aceptacion/'; 
if (!is_dir($directorio_cartas)) {
    if (!mkdir($directorio_cartas, 0775, true)) { 
        $_SESSION['error_message'] = "Error crítico: No se pudo crear el directorio para las cartas de aceptación.";
        error_log("PROCESAR_ACEPTACION: Falla al crear directorio: " . $directorio_cartas);
        header("Location: dashboard.php");
        exit;
    }
}
if (!is_writable($directorio_cartas)) {
    $_SESSION['error_message'] = "Error crítico: El directorio para guardar las cartas ('{$directorio_cartas}') no tiene permisos de escritura.";
    error_log("PROCESAR_ACEPTACION: Directorio no escribible: " . $directorio_cartas);
    header("Location: dashboard.php");
    exit;
}

$errores_proceso = [];
$cartas_generadas_count = 0;

foreach ($archivos_nombres_recibos as $nombre_recibo_original) {
    $nombre_recibo_sanitizado = basename(filter_var($nombre_recibo_original, FILTER_SANITIZE_SPECIAL_CHARS));
    if (empty($nombre_recibo_sanitizado)) {
        error_log("PROCESAR_ACEPTACION: Nombre de recibo inválido omitido: " . $nombre_recibo_original);
        $errores_proceso[] = "Se omitió un archivo de recibo con nombre inválido: " . htmlspecialchars($nombre_recibo_original);
        continue;
    }
    $timestamp = date('YmdHis'); 
    $nombre_base_recibo = pathinfo($nombre_recibo_sanitizado, PATHINFO_FILENAME);
    $nombre_archivo_carta = "carta_aceptacion_uid{$user_id_actual}_eid{$id_empleado_nomina_original}_{$nombre_base_recibo}_{$timestamp}.pdf";
    $ruta_completa_carta = $directorio_cartas . $nombre_archivo_carta;
    $parts = explode('_', $nombre_recibo_sanitizado);
    $descripcion_periodo = "Periodo no determinado";
    if (count($parts) > 4) { 
        $tipo_p = trim($parts[2] ?? 'Desconocido');
        $anio_p = (int)($parts[3] ?? 0);
        $num_p = (int)($parts[4] ?? 0);
        $descripcion_periodo = obtenerDescripcionPeriodo($tipo_p, $num_p, $anio_p);
    }

    $pdf = new FPDF('P', 'mm', 'Letter'); 
    $pdf->AddPage();
    $pdf->SetMargins(20, 20, 20);
    $pdf->SetAutoPageBreak(true, 25); 
    if (file_exists($ruta_logo_cliente_fpdf)) {
        list($ancho_logo, $alto_logo) = getimagesize($ruta_logo_cliente_fpdf);
        $ancho_max_logo = 60; 
        $alto_calculado_logo = ($ancho_logo > 0) ? (($ancho_max_logo / $ancho_logo) * $alto_logo) : 20;
        $pos_x_logo = ($pdf->GetPageWidth() - $ancho_max_logo) / 2; 
        $pdf->Image($ruta_logo_cliente_fpdf, $pos_x_logo, 15, $ancho_max_logo, 0); 
        $pdf->Ln($alto_calculado_logo + 5); 
    } else {
        error_log("PROCESAR_ACEPTACION: Logo para FPDF no encontrado en: " . $ruta_logo_cliente_fpdf);
        $pdf->Ln(15); 
    }
    $pdf->SetFont('Arial', 'B', 14);
    $pdf->Cell(0, 10, utf8_decode($nombre_empresa_cliente), 0, 1, 'C');
    $pdf->Ln(8);
    $pdf->SetFont('Arial', '', 11);
    $fecha_generacion_carta = date('d \d\e F \d\e Y'); 
    $pdf->MultiCell(0, 7, utf8_decode("A QUIEN CORRESPONDA:"), 0, 'L');
    $pdf->Ln(5);
    $texto_principal = "Por medio de la presente, se hace constar que el/la trabajador(a) C. " . $nombre_empleado . 
                       ", con ID de empleado N° " . $id_empleado_nomina_original . 
                       ", acepta y reconoce que la empresa " . $nombre_empresa_cliente . 
                       " le ha hecho entrega en este acto, en formato digital a través de la plataforma Nom-Online, ".
                       "del documento oficial denominado \"Recibo de Nómina\" correspondiente al periodo: " . $descripcion_periodo . ".";
    $pdf->MultiCell(0, 7, utf8_decode($texto_principal), 0, 'J'); 
    $pdf->Ln(5);
    $pdf->MultiCell(0, 7, utf8_decode("El/la trabajador(a) manifiesta su total conformidad y aceptación con la entrega de dicho documento, el cual ha sido descargado por él/ella mismo(a) desde la plataforma Nom-Online en esta fecha, sirviendo la presente como el más amplio acuse de recibo que en derecho proceda."), 0, 'J');
    $pdf->Ln(10);
    $ciudad_expedicion = "Ciudad de México"; 
    $pdf->Cell(0, 7, utf8_decode($ciudad_expedicion . ", a " . $fecha_generacion_carta . "."), 0, 1, 'R'); 
    $pdf->Ln(15);
    $pdf->SetFont('Arial', 'B', 11);
    $pdf->Cell(0, 7, utf8_decode("ATENTAMENTE"), 0, 1, 'C');
    $pdf->Ln(15); 
    $pdf->SetFont('Arial', '', 11);
    $pdf->Cell(0, 7, utf8_decode("___________________________________"), 0, 1, 'C');
    $pdf->Cell(0, 7, utf8_decode("C. " . $nombre_empleado), 0, 1, 'C');
    $pdf->Cell(0, 7, utf8_decode("ID de Empleado: " . $id_empleado_nomina_original), 0, 1, 'C');
    $pdf->Ln(10);
    $pdf->SetFont('Arial', 'I', 8);
    $pdf->Cell(0, 5, utf8_decode("Documento generado y aceptado electrónicamente el " . date('d/m/Y H:i:s') . " mediante la plataforma Nom-Online."), 0, 1, 'C');
    try {
        $pdf->Output('F', $ruta_completa_carta); 
        $cartas_generadas_count++;
    } catch (Exception $e) {
        $msg_error_pdf = "Error crítico al generar el PDF para el recibo '" . htmlspecialchars($nombre_recibo_sanitizado) . "': " . $e->getMessage();
        $_SESSION['error_message'] = ($_SESSION['error_message'] ?? '') . $msg_error_pdf . "<br>"; 
        error_log("PROCESAR_ACEPTACION: FPDF Output Exception: " . $e->getMessage() . " para archivo " . $ruta_completa_carta);
        $errores_proceso[] = $msg_error_pdf;
        continue; 
    }

    if ($conn) {
        $sql_insert_update = "INSERT INTO descargas_recibos (id_usuario, nombre_archivo_recibo, fecha_descarga, ruta_carta_aceptacion) 
                              VALUES (?, ?, NOW(), ?)
                              ON DUPLICATE KEY UPDATE 
                              fecha_descarga = NOW(), ruta_carta_aceptacion = VALUES(ruta_carta_aceptacion)";
        $stmt_insert_update = $conn->prepare($sql_insert_update);
        if ($stmt_insert_update) {
            $stmt_insert_update->bind_param("iss", $user_id_actual, $nombre_recibo_sanitizado, $nombre_archivo_carta); 
            if (!$stmt_insert_update->execute()) {
                $msg_error_db = "Error al registrar/actualizar la aceptación del recibo '" . htmlspecialchars($nombre_recibo_sanitizado) . "' en la base de datos: " . $stmt_insert_update->error;
                error_log("PROCESAR_ACEPTACION: DB Insert/Update Error: " . $stmt_insert_update->error);
                $errores_proceso[] = $msg_error_db;
            }
            $stmt_insert_update->close();
        } else {
            $msg_error_db_prep = "Error al preparar la consulta para registrar/actualizar la aceptación: " . $conn->error;
            error_log("PROCESAR_ACEPTACION: DB Prepare Insert/Update Error: " . $conn->error);
            $errores_proceso[] = $msg_error_db_prep;
        }
    } else {
        $msg_error_nodb = "No hay conexión a la base de datos para registrar la aceptación.";
        error_log("PROCESAR_ACEPTACION: " . $msg_error_nodb);
        $errores_proceso[] = $msg_error_nodb;
    }
} 

if (!empty($errores_proceso)) {
    $_SESSION['error_message'] = "Ocurrieron los siguientes errores durante el proceso de aceptación:<br>" . implode("<br>", array_map('htmlspecialchars', $errores_proceso));
} elseif ($cartas_generadas_count < count($archivos_nombres_recibos)) {
    $_SESSION['info_message'] = "Algunas cartas de aceptación se generaron. " . (count($archivos_nombres_recibos) - $cartas_generadas_count) . " archivo(s) no pudieron ser procesados completamente.";
    if ($tipo_descarga == 'individual' && $cartas_generadas_count > 0) { 
         $_SESSION['initiate_download_individual'] = $param_descarga_filename;
    } elseif ($tipo_descarga == 'masiva' && $cartas_generadas_count > 0) { 
         $_SESSION['initiate_download_bulk'] = [
            'filter_month_from' => $param_descarga_filter_month_from,
            'filter_month_to' => $param_descarga_filter_month_to,
        ];
    }
} else {
    $_SESSION['success_message'] = "¡Proceso de aceptación completado! Tu descarga comenzará en breve.";
    if ($tipo_descarga == 'individual') {
        if ($param_descarga_filename && in_array($param_descarga_filename, $archivos_nombres_recibos)) {
            $_SESSION['initiate_download_individual'] = $param_descarga_filename;
        } else {
            $_SESSION['error_message'] = ($_SESSION['error_message'] ?? '') . "<br>Error: El archivo individual para descarga no pudo ser verificado tras la aceptación.";
            error_log("PROCESAR_ACEPTACION: Discrepancia en filename individual. Esperado: {$param_descarga_filename}. Procesados: " . implode(', ',$archivos_nombres_recibos));
        }
    } elseif ($tipo_descarga == 'masiva') {
         $_SESSION['initiate_download_bulk'] = [
            'filter_month_from' => $param_descarga_filter_month_from,
            'filter_month_to' => $param_descarga_filter_month_to,
        ];
    } else {
        $_SESSION['error_message'] = ($_SESSION['error_message'] ?? '') . "<br>Tipo de descarga no reconocido después del procesamiento.";
    }
}

if (isset($conn) && $conn instanceof mysqli) {
    $conn->close();
}

header("Location: dashboard.php");
exit;
?>