<?php
// download_carta.php

// 1. Incluir verificadores y configuraciones
require_once('includes/auth_check.php'); // Verifica que haya una sesión activa
require_once('config/database.php');   // Para posible validación contra BD si es necesario

// 2. Verificar permisos (Supervisor o Admin)
//    auth_check.php ya verifica que esté logueado. Aquí verificamos el rol.
$is_supervisor = isset($_SESSION['is_supervisor']) && $_SESSION['is_supervisor'] === true;
$is_admin = isset($_SESSION['is_admin']) && $_SESSION['is_admin'] === true;

if (!$is_supervisor && !$is_admin) {
    // Si no es supervisor ni admin, no debería estar aquí.
    // Podrías mostrar un error o redirigir.
    // Para simplificar, si auth_check pasó, asumimos que supervisor_check ya hizo su trabajo
    // si el enlace a este script solo se muestra a supervisores/admins.
    // Pero una doble verificación no está de más.
    $_SESSION['error_message'] = "Acceso no autorizado para descargar esta carta.";
    header("Location: dashboard.php"); // O a index.php
    exit;
}

// 3. Obtener y sanitizar parámetros GET
$nombre_archivo_carta = filter_input(INPUT_GET, 'carta', FILTER_SANITIZE_SPECIAL_CHARS);
$empleado_user_id_param = filter_input(INPUT_GET, 'uid', FILTER_VALIDATE_INT); // ID del usuario al que pertenece la carta

if (empty($nombre_archivo_carta) || empty($empleado_user_id_param)) {
    die("Error: Parámetros incompletos o inválidos para la descarga de la carta.");
}

// 4. Construir la ruta al archivo y verificar
//    Las cartas se guardan en 'cartas_aceptacion/' en la raíz del proyecto.
//    El nombre del archivo de la carta ya está en $nombre_archivo_carta.
$directorio_cartas = __DIR__ . '/cartas_aceptacion/';
$ruta_completa_carta = $directorio_cartas . basename($nombre_archivo_carta); // basename() para seguridad adicional

// Medida de seguridad adicional: Validar que la carta pertenece al UID proporcionado (Opcional, si el nombre del archivo no es suficientemente único o si se quiere más control)
// Esto requeriría una consulta a la BD para verificar que 'ruta_carta_aceptacion' = $nombre_archivo_carta y 'id_usuario' = $empleado_user_id_param
// Por ahora, asumimos que si el supervisor/admin llegó aquí con el nombre correcto, está bien.
// Si el nombre del archivo de la carta ya incluye el user_id (como sugerí: carta_aceptacion_uid{$user_id_actual}_...),
// podrías incluso verificar que el $empleado_user_id_param coincida con el uid en el nombre del archivo.

if (file_exists($ruta_completa_carta) && is_readable($ruta_completa_carta)) {
    // 5. Enviar cabeceras para la descarga del PDF
    header('Content-Type: application/pdf');
    header('Content-Disposition: inline; filename="' . basename($nombre_archivo_carta) . '"'); // 'inline' para intentar mostrar en navegador, 'attachment' para forzar descarga
    header('Content-Transfer-Encoding: binary');
    header('Accept-Ranges: bytes');
    header('Content-Length: ' . filesize($ruta_completa_carta)); // Importante para algunos navegadores/proxies

    // 6. Enviar el contenido del archivo
    ob_clean(); // Limpiar cualquier salida previa
    flush();    // Enviar salida al navegador
    readfile($ruta_completa_carta);
    exit;
} else {
    error_log("DOWNLOAD_CARTA: Archivo de carta no encontrado o no legible. Ruta: " . $ruta_completa_carta . " Solicitado por user_id: " . ($_SESSION['user_id'] ?? ' desconocido'));
    // Mostrar un error más amigable al usuario
    // header('HTTP/1.0 404 Not Found'); // Opcional
    echo "<!DOCTYPE html><html><head><title>Error</title><link rel='stylesheet' href='/public/css/estilos.css'></head><body>";
    echo "<main class='container'><p class='error'>Error: El archivo de la carta de aceptación no se encontró o no se puede leer. Por favor, contacte al administrador.</p>";
    echo "<p><a href='javascript:history.back()' class='button'>Regresar</a></p></main></body></html>";
    exit;
}
?>