<?php
// download.php

require_once('includes/auth_check.php'); // Necesita estar logueado
// *** AÑADIDO: Incluir la conexión a la base de datos ***
require_once('config/database.php');

// Obtener el ID de empleado de la sesión ACTUAL
$employee_id_session = $_SESSION['employee_id'] ?? null;
$is_admin = $_SESSION['is_admin'] ?? false;

// *** AÑADIDO: Obtener el ID de usuario de la sesión para registrar la descarga ***
$user_id_session = $_SESSION['user_id'] ?? null;

if ($employee_id_session === null && !$is_admin) {
    header('HTTP/1.0 403 Forbidden');
    // Es buena práctica no dar demasiados detalles al usuario en producción
    // error_log("Acceso denegado (sin ID de sesión) en download.php");
    exit("Acceso denegado.");
}

// 1. Obtener y Validar Nombre de Archivo
$filename = $_GET['filename'] ?? '';

if (empty($filename)) {
    header('HTTP/1.0 400 Bad Request');
    // error_log("Nombre de archivo no proporcionado en download.php");
    exit("Nombre de archivo no proporcionado.");
}

if (strpos($filename, '/') !== false || strpos($filename, '\\') !== false || strpos($filename, '..') !== false) {
    header('HTTP/1.0 400 Bad Request');
    // error_log("Intento de Path Traversal en download.php: " . $filename . " por user_id: " . ($user_id_session ?? 'N/A'));
    exit("Nombre de archivo inválido.");
}

// 2. Extraer ID y Comparar
$parts = explode('_', $filename);
$file_employee_id_str = null; // Cambiado nombre para claridad
if (count($parts) > 5 && isset($parts[5])) {
    // Extraer la parte del ID y limpiar cualquier extensión como .zip o .rar
    $id_part_raw = $parts[5];
    $file_employee_id_str = pathinfo($id_part_raw, PATHINFO_FILENAME);
}

$file_id_int = ($file_employee_id_str !== null && is_numeric(trim($file_employee_id_str))) ? (int)trim($file_employee_id_str) : null;
$session_id_int = ($employee_id_session !== null && is_numeric($employee_id_session)) ? (int)$employee_id_session : null;

if ($file_id_int === null || ($file_id_int !== $session_id_int && !$is_admin)) {
    error_log("DOWNLOAD FORBIDDEN: User {$user_id_session} (EmpID: {$session_id_int}) tried to download {$filename} (FileEmpID: {$file_id_int}, RawFilePart: {$file_employee_id_str})");
    header('HTTP/1.0 403 Forbidden');
    exit("Acceso denegado a este archivo.");
}

// 3. Construir Ruta y Verificar
$recibos_dir = __DIR__ . '/recibos_nomina';
$filePath = $recibos_dir . '/' . $filename;

if (!file_exists($filePath) || !is_readable($filePath)) {
    error_log("DOWNLOAD NOT FOUND: " . $filePath . " requested by User {$user_id_session}");
    header('HTTP/1.0 404 Not Found');
    exit("Archivo no encontrado en el servidor.");
}

// 4. Enviar Cabeceras y Archivo
header('Content-Description: File Transfer');
header('Content-Type: application/octet-stream');
header('Content-Disposition: attachment; filename="' . basename($filename) . '"'); // basename() es buena práctica aquí
header('Expires: 0');
header('Cache-Control: must-revalidate');
header('Pragma: public');
header('Content-Length: ' . filesize($filePath));
ob_clean();
flush();

// Enviar el archivo al navegador
$download_success = readfile($filePath);

if ($download_success !== false) {
    // --- INICIO: NUEVA LÓGICA PARA REGISTRAR LA DESCARGA ---
    if ($user_id_session !== null && !empty($filename) && $conn) { // Asegurarse que $conn y $user_id_session existan
        // Usamos INSERT IGNORE para evitar errores si el registro ya existe
        $sql_insert_descarga = "INSERT IGNORE INTO descargas_recibos (id_usuario, nombre_archivo_recibo, fecha_descarga) VALUES (?, ?, CURRENT_TIMESTAMP)";
        $stmt_insert = $conn->prepare($sql_insert_descarga);

        if ($stmt_insert) {
            $stmt_insert->bind_param("is", $user_id_session, $filename);
            if (!$stmt_insert->execute()) {
                error_log("Error al registrar descarga para user_id: $user_id_session, archivo: $filename - " . $stmt_insert->error);
            }
            $stmt_insert->close();
        } else {
            error_log("Error al preparar statement para registrar descarga (User: $user_id_session, File: $filename): " . $conn->error);
        }
    } elseif (!$conn) {
        error_log("Error de conexión a BD al intentar registrar descarga (User: $user_id_session, File: $filename)");
    }
    // --- FIN: NUEVA LÓGICA PARA REGISTRAR LA DESCARGA ---
} else {
    error_log("Error en readfile para: " . $filePath . " (User: $user_id_session)");
    // No envíes 'exit' aquí si readfile falla, ya que podría cortar un archivo parcialmente enviado.
    // El navegador generalmente detectará un error si la descarga es incompleta.
}

// Siempre cerrar la conexión si está abierta y luego salir.
if ($conn) {
    $conn->close();
}
exit; // Terminar el script después de intentar enviar el archivo y registrar la descarga
?>