<?php
// change_initial_password.php
if (session_status() === PHP_SESSION_NONE) {
    session_start([
        'cookie_httponly' => true,
        'cookie_secure' => isset($_SERVER['HTTPS']), 
        'cookie_samesite' => 'Lax'
    ]);
}

// 1. Verificar que el usuario deba estar aquí
if (!isset($_SESSION['user_id']) || !isset($_SESSION['must_change_password']) || $_SESSION['must_change_password'] !== true) {
    if (isset($_SESSION['user_id'])) {
        header("Location: index.php"); 
    } else {
        header("Location: login.php?error=auth_required"); 
    }
    exit;
}

require_once('config/database.php');
require_once('includes/functions.php'); // Para password_hash y funciones CSRF

$page_title = 'Establecer Nueva Contraseña';
$error_messages = [];

$password_requirements_html = "<ul>
    <li>Mínimo 8 caracteres de longitud.</li>
    <li>Al menos una letra mayúscula (A-Z).</li>
    <li>Al menos una letra minúscula (a-z).</li>
    <li>Al menos un número (0-9).</li>
    <li>Al menos un carácter especial (ej. !@#$%^&*()-_=+<>?).</li>
</ul>";

if (!function_exists('validarPasswordConRequisitos')) {
    function validarPasswordConRequisitos($password) {
        $errors = [];
        if (strlen($password) < 8) { $errors[] = "La contraseña debe tener al menos 8 caracteres.";}
        if (!preg_match('/[A-Z]/', $password)) { $errors[] = "La contraseña debe contener al menos una letra mayúscula.";}
        if (!preg_match('/[a-z]/', $password)) { $errors[] = "La contraseña debe contener al menos una letra minúscula.";}
        if (!preg_match('/[0-9]/', $password)) { $errors[] = "La contraseña debe contener al menos un número.";}
        if (!preg_match('/[!@#$%^&*()\-_=+<>?]/', $password)) { $errors[] = "La contraseña debe contener al menos un carácter especial.";}
        return $errors;
    }
}

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // VALIDAR TOKEN CSRF PRIMERO
    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        $error_messages[] = "Error de validación de seguridad. Por favor, intente de nuevo.";
        error_log("CSRF token validation failed for change_initial_password.php. User ID: " . ($_SESSION['user_id'] ?? 'N/A'));
        // No procesar nada más si el token es inválido
    } else {
        // Token válido, continuar con el procesamiento de la contraseña
        $new_password = $_POST['new_password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';

        if (empty($new_password) || empty($confirm_password)) {
            $error_messages[] = "Ambos campos de contraseña son obligatorios.";
        } elseif ($new_password !== $confirm_password) {
            $error_messages[] = "Las contraseñas no coinciden.";
        } else {
            $complexity_errors = validarPasswordConRequisitos($new_password);
            if (!empty($complexity_errors)) {
                $error_messages = array_merge($error_messages, $complexity_errors);
            }
        }

        if (empty($error_messages)) { // Solo proceder si no hay errores de contraseña NI de CSRF
            $user_id_to_update = $_SESSION['user_id'];
            $new_password_hash = password_hash($new_password, PASSWORD_DEFAULT);

            // Asegurarse que $conn esté disponible si no se incluyó antes del bloque POST
            if (!isset($conn) || !$conn) { // Esto es una doble verificación, config/database.php ya debería haberlo hecho.
                 require('config/database.php'); // Intentar reconectar si es necesario
            }

            if ($conn) {
                $sql = "UPDATE user SET password_hash = ?, force_password_change = 0 WHERE id = ?";
                $stmt = $conn->prepare($sql);
                if ($stmt) {
                    $stmt->bind_param("si", $new_password_hash, $user_id_to_update);
                    if ($stmt->execute()) {
                        unset($_SESSION['must_change_password']);
                        
                        session_unset(); 
                        session_destroy(); 

                        if (session_status() === PHP_SESSION_NONE) {
                            session_start([
                                'cookie_httponly' => true,
                                'cookie_secure' => isset($_SERVER['HTTPS']),
                                'cookie_samesite' => 'Lax'
                            ]);
                        }
                        header("Location: login.php?message=password_changed");
                        exit;

                    } else {
                        $error_messages[] = "Error al actualizar la contraseña en la base de datos. Por favor, inténtelo de nuevo.";
                        error_log("CHANGE_INITIAL_PASSWORD: Error al ejecutar update: " . $stmt->error);
                    }
                    $stmt->close();
                } else {
                    $error_messages[] = "Error del servidor al preparar la actualización. Por favor, inténtelo más tarde.";
                    error_log("CHANGE_INITIAL_PASSWORD: Error al preparar update: " . $conn->error);
                }
            } else {
                 $error_messages[] = "Error de conexión a la base de datos.";
                 error_log("CHANGE_INITIAL_PASSWORD: No hay conexión a BD en POST.");
            }
        }
    } 
    // Cerrar la conexión si se abrió dentro del bloque POST y hubo error antes de redirigir
    if(isset($conn) && $conn instanceof mysqli && $_SERVER["REQUEST_METHOD"] == "POST" && !empty($error_messages)) {
        $conn->close();
    }
}

// Generar token CSRF para el formulario (se genera siempre, incluso si hay errores POST para el siguiente intento)
$csrf_token = generate_csrf_token();

require_once('includes/header.php'); 
?>
<main class="container">
    <h2><?php echo htmlspecialchars($page_title); ?></h2>

    <?php if (!empty($error_messages)): ?>
        <div class="error" style="margin-bottom: 15px;">
            <strong>Por favor, corrija los siguientes errores:</strong><br>
            <?php foreach ($error_messages as $error): ?>
                - <?php echo htmlspecialchars($error); ?><br>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>

    <p>Debido a que esta es la primera vez que inicia sesión o su contraseña ha sido reseteada, debe establecer una nueva contraseña personal.</p>
    
    <div style="margin-bottom: 20px; padding: 10px; border: 1px solid #ddd; background-color: #f9f9f9; border-radius:4px;">
        <strong>Requisitos para la nueva contraseña:</strong>
        <?php echo $password_requirements_html; ?>
    </div>

    <form action="change_initial_password.php" method="POST">
        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">

        <div class="form-group">
            <label for="new_password">Nueva Contraseña:</label>
            <input type="password" id="new_password" name="new_password" required>
        </div>
        <div class="form-group">
            <label for="confirm_password">Confirmar Nueva Contraseña:</label>
            <input type="password" id="confirm_password" name="confirm_password" required>
        </div>
        <div class="form-group">
            <button type="submit" class="button">Establecer Nueva Contraseña</button>
        </div>
    </form>

</main>
<?php
require_once('includes/footer.php'); 
// Asegurar que la conexión se cierre si se abrió fuera del bloque POST o si el bloque POST no se ejecutó.
if (isset($conn) && $conn instanceof mysqli && $conn->ping()) { 
    $conn->close();
}
?>