<?php
// admin/user_delete.php

require_once('../includes/admin_check.php'); // Verifica que sea admin
require_once('../config/database.php');   // Conexión a la BD
require_once('../includes/functions.php'); // Para validate_csrf_token()

// Inicializar mensajes para la redirección
$_SESSION['error_message'] = null;
$_SESSION['success_message'] = null;

// Verificar que la solicitud sea POST
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // 1. Validar el token CSRF
    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        $_SESSION['error_message'] = "Error de validación de seguridad (CSRF token inválido). La acción no se realizó.";
        header("Location: users_list.php");
        exit;
    }

    // 2. Obtener el ID del usuario del POST
    $user_id_to_delete = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
    $error_flag = false; // Usaremos una bandera para los errores

    if ($user_id_to_delete) {
        // 3. Prevenir borrado del propio admin logueado
        if ($user_id_to_delete == $_SESSION['user_id']) {
            $_SESSION['error_message'] = "No puedes eliminar tu propia cuenta de administrador.";
            $error_flag = true;
        } else {
            // Podrías añadir una verificación para no borrar el último admin si solo queda uno,
            // pero eso es una lógica más compleja. Por ahora, solo prevenimos el auto-borrado.

            // 4. Proceder con la eliminación
            $sql = "DELETE FROM user WHERE id = ?";
            $stmt = $conn->prepare($sql);
            if ($stmt) {
                $stmt->bind_param("i", $user_id_to_delete);
                if ($stmt->execute()) {
                    if ($stmt->affected_rows > 0) {
                        $_SESSION['success_message'] = "Usuario eliminado correctamente.";
                    } else {
                        // Esto podría pasar si el usuario ya fue eliminado por otra acción
                        // o si el ID no existía (aunque el enlace viene de una lista existente).
                        $_SESSION['error_message'] = "Usuario no encontrado o ya había sido eliminado.";
                        $error_flag = true; 
                    }
                } else {
                    $_SESSION['error_message'] = "Error al ejecutar la eliminación del usuario: " . $stmt->error;
                    error_log("USER_DELETE (Execute Error): " . $stmt->error . " para ID: " . $user_id_to_delete);
                    $error_flag = true;
                }
                $stmt->close();
            } else {
                $_SESSION['error_message'] = "Error preparando la consulta de eliminación: " . $conn->error;
                error_log("USER_DELETE (Prepare Error): " . $conn->error);
                $error_flag = true;
            }
        }
    } else {
        $_SESSION['error_message'] = "No se proporcionó un ID de usuario válido para eliminar.";
        $error_flag = true;
    }

    if (isset($conn) && $conn instanceof mysqli) {
        $conn->close();
    }

} else {
    // Si no es una solicitud POST, es un acceso inválido a este script
    $_SESSION['error_message'] = "Acceso no válido para eliminar usuario.";
}

// Redirigir siempre de vuelta a la lista de usuarios
header("Location: users_list.php");
exit;
?>