<?php
// admin/reset_send_access.php

require_once('../includes/admin_check.php'); 
require_once('../config/database.php');
require_once('../includes/functions.php'); 

$_SESSION['error_message'] = null;
$_SESSION['success_message'] = null;
$_SESSION['reset_warning_message'] = null; // Para mensajes de advertencia

$url_del_sitio = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://" . $_SERVER['HTTP_HOST'];
$url_del_sitio .= "/login.php"; 

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        $_SESSION['error_message'] = "Error de validación de seguridad (CSRF token inválido). La acción no se realizó.";
        header("Location: users_list.php");
        exit;
    }

    $user_id_to_reset = filter_input(INPUT_POST, 'user_id_to_reset', FILTER_VALIDATE_INT);

    if (empty($user_id_to_reset)) {
        $_SESSION['error_message'] = "No se especificó un ID de usuario válido.";
        header("Location: users_list.php");
        exit;
    }

    // Iniciar transacción
    $conn->begin_transaction();

    try {
        // Obtener datos del usuario (username, full_name, email, initial_access_sent)
        $stmt_user = $conn->prepare("SELECT username, full_name, email, initial_access_sent FROM user WHERE id = ?");
        if (!$stmt_user) throw new Exception("Error preparando consulta de usuario: " . $conn->error);
        
        $stmt_user->bind_param("i", $user_id_to_reset);
        $stmt_user->execute();
        $result_user = $stmt_user->get_result();

        if ($user_data = $result_user->fetch_assoc()) {
            $username = $user_data['username'];
            $full_name = $user_data['full_name'];
            $email_destinatario = $user_data['email'];
            $was_initial_access_sent = (bool)$user_data['initial_access_sent'];

            if (empty($email_destinatario)) {
                $_SESSION['error_message'] = "El usuario '" . htmlspecialchars($username) . "' no tiene un correo electrónico configurado para enviar los accesos.";
                $conn->rollback(); // No hay nada que hacer si no hay email
                header("Location: users_list.php");
                exit;
            }

            $nueva_password_plain = generarPasswordSeguro(12);
            $nuevo_password_hash = password_hash($nueva_password_plain, PASSWORD_DEFAULT);
            $force_change = 1; 
            $new_initial_access_sent_value = 1; // Siempre se marcará como enviado después de esta acción.

            $sql_update = "UPDATE user SET password_hash = ?, force_password_change = ?, initial_access_sent = ? WHERE id = ?";
            $stmt_update = $conn->prepare($sql_update);
            if (!$stmt_update) throw new Exception("Error preparando update de contraseña: " . $conn->error);
            
            $stmt_update->bind_param("siii", $nuevo_password_hash, $force_change, $new_initial_access_sent_value, $user_id_to_reset);
            
            if ($stmt_update->execute()) {
                if (enviarCorreoAccesos($email_destinatario, $full_name, $username, $nueva_password_plain, $url_del_sitio)) {
                    if (!$was_initial_access_sent) {
                        $_SESSION['success_message'] = "Se han enviado los ACCESOS INICIALES por correo a '" . htmlspecialchars($username) . "' (" . htmlspecialchars($email_destinatario) . ") con una nueva contraseña temporal.";
                    } else {
                        $_SESSION['success_message'] = "Se han REENVIADO los accesos (con una nueva contraseña temporal) por correo a '" . htmlspecialchars($username) . "' (" . htmlspecialchars($email_destinatario) . ").";
                    }
                } else {
                    // El correo falló, pero la contraseña se reseteó. Informar al admin.
                    $_SESSION['error_message'] = "Se generó una nueva contraseña para '" . htmlspecialchars($username) . "', pero falló el envío del correo a " . htmlspecialchars($email_destinatario) . ".";
                    $_SESSION['reset_warning_message'] = "Contraseña para " . htmlspecialchars($username) . ": <strong>" . htmlspecialchars($nueva_password_plain) . "</strong> (Por favor, comuníquela manualmente).";
                }
                $conn->commit();
            } else {
                throw new Exception("Error al actualizar la contraseña del usuario '" . htmlspecialchars($username) . "': " . $stmt_update->error);
            }
            $stmt_update->close();
        } else {
            $_SESSION['error_message'] = "Usuario con ID " . htmlspecialchars($user_id_to_reset) . " no encontrado.";
            $conn->rollback(); // Aunque no hubo cambios, es buena práctica
        }
        $stmt_user->close();
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error_message'] = "Error en el proceso: " . $e->getMessage();
        error_log("RESET_SEND_ACCESS: Excepción: " . $e->getMessage());
    }
    
    if (isset($conn) && $conn instanceof mysqli) $conn->close();

} else {
    $_SESSION['error_message'] = "Acceso no válido.";
}

header("Location: users_list.php");
exit;
?>