<?php
// admin/bulk_initial_access.php

require_once('../includes/admin_check.php');
require_once('../config/database.php');
require_once('../includes/functions.php');

$_SESSION['error_message'] = null;
$_SESSION['success_message'] = null;
$_SESSION['info_message'] = null; // Usaremos este para el resumen detallado

// URL del sitio para el correo
$url_del_sitio = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://" . $_SERVER['HTTP_HOST'];
$url_del_sitio .= "/login.php"; 

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (!isset($_POST['csrf_token']) || !validate_csrf_token($_POST['csrf_token'])) {
        $_SESSION['error_message'] = "Error de validación de seguridad (CSRF). La acción no se realizó.";
        header("Location: users_list.php");
        exit;
    }

    $current_page_from_post = filter_input(INPUT_POST, 'current_page', FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);
    $items_per_page_from_post = filter_input(INPUT_POST, 'items_per_page', FILTER_VALIDATE_INT, ['options' => ['default' => 10, 'min_range' => 1]]);
    
    $offset = ($current_page_from_post - 1) * $items_per_page_from_post;

    // Obtener usuarios de la página actual que NO hayan recibido accesos iniciales Y TENGAN EMAIL
    $sql_users_to_notify = "SELECT id, username, full_name, email 
                            FROM user 
                            WHERE initial_access_sent = 0 AND email IS NOT NULL AND email <> ''
                            ORDER BY username 
                            LIMIT ? OFFSET ?";
    
    $stmt_users = $conn->prepare($sql_users_to_notify);
    if (!$stmt_users) {
        $_SESSION['error_message'] = "Error al preparar la consulta de usuarios para notificar: " . $conn->error;
        error_log("BULK_INITIAL_ACCESS: Prepare error: " . $conn->error);
        header("Location: users_list.php?page=" . $current_page_from_post);
        exit;
    }

    $stmt_users->bind_param("ii", $items_per_page_from_post, $offset);
    $stmt_users->execute();
    $result_users = $stmt_users->get_result();

    $processed_count = 0;
    $sent_success_count = 0;
    $sent_fail_count = 0;
    $details_for_admin = []; // Para contraseñas si el correo falla

    if ($result_users->num_rows > 0) {
        $conn->begin_transaction();
        try {
            while ($user = $result_users->fetch_assoc()) {
                $user_id = $user['id'];
                $username = $user['username'];
                $full_name = $user['full_name'];
                $email_destinatario = $user['email'];

                $nueva_password_plain = generarPasswordSeguro(12);
                $nuevo_password_hash = password_hash($nueva_password_plain, PASSWORD_DEFAULT);
                $force_change = 1;
                $set_initial_access_sent = 1;

                $stmt_update = $conn->prepare("UPDATE user SET password_hash = ?, force_password_change = ?, initial_access_sent = ? WHERE id = ?");
                if (!$stmt_update) throw new Exception("Error preparando update para usuario {$username}: " . $conn->error);
                
                $stmt_update->bind_param("siii", $nuevo_password_hash, $force_change, $set_initial_access_sent, $user_id);
                
                if ($stmt_update->execute()) {
                    $processed_count++;
                    if (enviarCorreoAccesos($email_destinatario, $full_name, $username, $nueva_password_plain, $url_del_sitio)) {
                        $sent_success_count++;
                    } else {
                        $sent_fail_count++;
                        $details_for_admin[] = "Usuario: " . htmlspecialchars($username) . " - Email: " . htmlspecialchars($email_destinatario) . " - Contraseña generada (envío falló): " . htmlspecialchars($nueva_password_plain);
                    }
                } else {
                    throw new Exception("Error actualizando usuario {$username}: " . $stmt_update->error);
                }
                $stmt_update->close();
            }
            $conn->commit();
            
            $summary_message = "Proceso de envío masivo de accesos iniciales completado para la página actual.<br>";
            $summary_message .= "Usuarios elegibles procesados: {$processed_count}.<br>";
            $summary_message .= "Correos enviados exitosamente: {$sent_success_count}.<br>";
            if ($sent_fail_count > 0) {
                $summary_message .= "Correos fallidos: {$sent_fail_count}.<br>";
                if (!empty($details_for_admin)) {
                    $summary_message .= "<strong>IMPORTANTE - Contraseñas para usuarios con envío fallido (comunicar manualmente):</strong><br>";
                    foreach($details_for_admin as $detail) {
                        $summary_message .= $detail . "<br>";
                    }
                }
            }
            if ($processed_count == 0) {
                $summary_message = "No había usuarios elegibles (con email y sin accesos enviados previamente) en la página actual para procesar.";
            }
            $_SESSION['info_message'] = $summary_message;

        } catch (Exception $e) {
            $conn->rollback();
            $_SESSION['error_message'] = "Ocurrió un error durante el proceso masivo: " . $e->getMessage();
            error_log("BULK_INITIAL_ACCESS: Exception: " . $e->getMessage());
        }
    } else {
        $_SESSION['info_message'] = "No se encontraron usuarios en la página actual que requieran el envío de accesos iniciales (o no tienen email registrado).";
    }
    $stmt_users->close();

} else {
    $_SESSION['error_message'] = "Acceso no válido para esta operación.";
}

if (isset($conn) && $conn instanceof mysqli) $conn->close();
header("Location: users_list.php?page=" . $current_page_from_post); // Volver a la misma página de la lista
exit;
?>