<?php
// admin/bulk_download.php

// 1. Incluir archivos necesarios
require_once('../includes/auth_check.php'); // ¡Fundamental para seguridad y obtener employee_id!
require_once('../includes/functions.php'); // Para obtenerNumeroMes()

// 2. Verificar que se accede por POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['bulk_download_error'] = "Acceso no permitido.";
    header('Location: ../dashboard.php');
    exit;
}

// 3. Leer parámetros de filtro desde POST (formato YYYY-MM)
$filter_month_from_str = filter_input(INPUT_POST, 'filter_month_from', FILTER_SANITIZE_SPECIAL_CHARS);
$filter_month_to_str = filter_input(INPUT_POST, 'filter_month_to', FILTER_SANITIZE_SPECIAL_CHARS);

// Parsear las fechas de inicio y fin (igual que en dashboard.php)
$start_year = null; $start_month = null; $start_year_month = null;
if ($filter_month_from_str && preg_match('/^(\d{4})-(\d{2})$/', $filter_month_from_str, $matches_from)) {
    $start_year = (int)$matches_from[1];
    $start_month = (int)$matches_from[2];
    $start_year_month = $start_year * 100 + $start_month; // ej: 202403
}

$end_year = null; $end_month = null; $end_year_month = null;
if ($filter_month_to_str && preg_match('/^(\d{4})-(\d{2})$/', $filter_month_to_str, $matches_to)) {
    $end_year = (int)$matches_to[1];
    $end_month = (int)$matches_to[2];
    $end_year_month = $end_year * 100 + $end_month; // ej: 202404
}

// Validar rango
if ($start_year_month !== null && $end_year_month !== null && $start_year_month > $end_year_month) {
    $_SESSION['bulk_download_error'] = "El rango de fechas para la descarga es inválido (Mes Desde no puede ser posterior a Mes Hasta).";
    header('Location: ../dashboard.php');
    exit;
}

// 4. Obtener employee_id de la sesión
$employee_id = $_SESSION['employee_id'] ?? null;

if ($employee_id === null) {
    $_SESSION['bulk_download_error'] = "Error: No se pudo verificar tu identidad para la descarga.";
    header('Location: ../dashboard.php');
    exit;
}

// 5. Lógica de Filtrado de Archivos
$filtered_files_paths = [];
$recibos_dir = __DIR__ . '/../recibos_nomina';

if (!is_dir($recibos_dir) || !is_readable($recibos_dir)) {
    $_SESSION['bulk_download_error'] = "Error: No se puede acceder a la carpeta de recibos para la descarga.";
    error_log("Error crítico en bulk_download: La carpeta de recibos no existe o no es legible: " . $recibos_dir);
    header('Location: ../dashboard.php');
    exit;
}

$files = glob($recibos_dir . "/*.{zip,rar,ZIP,RAR}", GLOB_BRACE);

if ($files === false) {
    $_SESSION['bulk_download_error'] = "Error al intentar leer los archivos de recibos para la descarga.";
    error_log("Error en glob() para bulk_download: " . $recibos_dir);
    header('Location: ../dashboard.php');
    exit;
}

if (!empty($files)) {
    foreach ($files as $filePath) {
        $filename = basename($filePath);
        $parts = explode('_', $filename);

        if (count($parts) > 5 && isset($parts[5]) && (int)trim($parts[5]) == (int)$employee_id) {
            $tipoPeriodo = trim($parts[2] ?? 'Desconocido');
            $anio_file = (int)($parts[3] ?? 0);
            $numeroPeriodo_file = (int)($parts[4] ?? 0);
            $mes_archivo = obtenerNumeroMes($tipoPeriodo, $numeroPeriodo_file);

            $pasarFiltro = true;

            // Lógica de filtrado por mes/año (idéntica a dashboard.php)
            if ($mes_archivo === null && ($start_year_month !== null || $end_year_month !== null)) {
                $pasarFiltro = false;
            } elseif ($mes_archivo !== null) {
                $file_year_month = $anio_file * 100 + $mes_archivo;
                if ($start_year_month !== null && $file_year_month < $start_year_month) {
                    $pasarFiltro = false;
                }
                if ($end_year_month !== null && $file_year_month > $end_year_month) {
                    $pasarFiltro = false;
                }
            }
            // Si no hay ningún filtro de mes/año activo Y este archivo no tiene mes (o sí lo tiene),
            // se incluiría. Pero el botón de descarga solo aparece si $filters_active es true,
            // lo que implica que al menos un filtro de mes/año está presente.
            // Si ambos $start_year_month y $end_year_month son null aquí, significa que
            // el usuario no seleccionó nada en los filtros, pero de alguna manera el form se envió.
            // En ese caso, $pasarFiltro seguiría true y se descargarían todos los archivos del empleado.
            // Sin embargo, el botón en dashboard.php solo aparece si $filters_active = true,
            // lo que significa que al menos $filter_month_from_str o $filter_month_to_str tienen valor.
            // Por lo tanto, o $start_year_month o $end_year_month (o ambos) tendrán valor aquí.


            if ($pasarFiltro) {
                $filtered_files_paths[] = $filePath;
            }
        }
    }
}

// 6. Creación y Envío del ZIP
if (!empty($filtered_files_paths)) {
    $zip = new ZipArchive();
    $zip_filename = sys_get_temp_dir() . '/recibos_filtrados_EMP' . $employee_id . '_' . time() . '.zip';

    if ($zip->open($zip_filename, ZipArchive::CREATE | ZipArchive::OVERWRITE) === TRUE) {
        foreach ($filtered_files_paths as $file_to_add) {
            if (file_exists($file_to_add)) {
               $zip->addFile($file_to_add, basename($file_to_add));
            }
        }
        $zip->close();

        if (file_exists($zip_filename)) {
            header('Content-Type: application/zip');
            header('Content-Disposition: attachment; filename="' . basename($zip_filename) . '"');
            header('Content-Length: ' . filesize($zip_filename));
            header('Cache-Control: no-cache, no-store, must-revalidate'); // HTTP 1.1.
            header('Pragma: no-cache'); // HTTP 1.0.
            header('Expires: 0'); // Proxies.
            ob_clean();
            flush();
            readfile($zip_filename);
            unlink($zip_filename);
            exit;
        } else {
            $_SESSION['bulk_download_error'] = "Error: No se pudo crear el archivo ZIP de descarga.";
            error_log("Error en bulk_download: ZipArchive cerrado pero el archivo ZIP no existe: $zip_filename");
        }
    } else {
        $_SESSION['bulk_download_error'] = "Error: No se pudo inicializar la creación del archivo ZIP.";
        error_log("Error en bulk_download: ZipArchive::open falló para $zip_filename");
    }
} else {
    $_SESSION['bulk_download_info'] = "No se encontraron recibos que coincidan con los filtros seleccionados para la descarga.";
}

// 7. Redirigir de vuelta al Dashboard si no se envió un archivo
header('Location: ../dashboard.php');
exit;
?>